package de.panmobil.scompdemo;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
//import java.util.Arrays;
import java.util.UUID;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothSocket;
import android.content.Context;
import android.os.Bundle;
//import android.os.Debug;
import android.os.Handler;
import android.os.Message;
import android.util.Log;
//import android.util.Log;

public class BluetoothScanndyService {
    private static final UUID SPP_UUID = UUID.fromString("00001101-0000-1000-8000-00805f9b34fb");

    private final BluetoothAdapter mAdapter;
    private final Handler mHandler;
    private ConnectThread mConnectThread;
    private ConnectedThread mConnectedThread;
    private int mState;

    public static final int STATE_NONE = 0;       // we're doing nothing
    public static final int STATE_CONNECTING = 2; // now initiating an outgoing connection
    public static final int STATE_CONNECTED = 3;  // now connected to a remote device

	// Message types sent
    public static final int MESSAGE_STATE_CHANGE = 1;
    public static final int MESSAGE_DEVICE_NAME = 2;
    public static final int MESSAGE_CONNECTION_FAILED = 3;
    public static final int MESSAGE_CONNECTION_LOST = 4;
    public static final int MESSAGE_READ = 5;
    public static final int MESSAGE_WRITE = 6;

    // MESSAGE_DEVICE_NAME bundle key
    public static final String DEVICE_NAME = "device_name";
    
    public static final String SCOMP = "scomp";

    public BluetoothScanndyService(Context context, Handler handler) {
        mAdapter = BluetoothAdapter.getDefaultAdapter();
        mState = STATE_NONE;
        mHandler = handler;
    }

    private synchronized void setState(int state) {
        mState = state;

        mHandler.obtainMessage(MESSAGE_STATE_CHANGE, state, -1).sendToTarget();
    }

    public synchronized int getState() {
        return mState;
    }
    
    private void deleteConnectThread() {
        if (mConnectThread != null) {
        	mConnectThread.cancel();
        	mConnectThread = null;
    	}
    }

    private void deleteConnectedThread() {
        if (mConnectedThread != null) {
        	mConnectedThread.cancel();
        	mConnectedThread = null;
    	}
    }

    public synchronized void start() {
        deleteConnectThread();
        deleteConnectedThread();

        setState(STATE_NONE);
    }

    public synchronized void connect(BluetoothDevice device) {
        if (mState == STATE_CONNECTING)
        	deleteConnectThread();

        deleteConnectedThread();

        mConnectThread = new ConnectThread(device);
        mConnectThread.start();
        setState(STATE_CONNECTING);
    }

    public synchronized void connected(BluetoothSocket socket, BluetoothDevice device) {
        deleteConnectThread();
        deleteConnectedThread();

        mConnectedThread = new ConnectedThread(socket);
        mConnectedThread.start();

        // Send the name of the connected device back to the UI Activity
        Message msg = mHandler.obtainMessage(MESSAGE_DEVICE_NAME);
        Bundle bundle = new Bundle();
        bundle.putString(DEVICE_NAME, device.getName());
        msg.setData(bundle);
        mHandler.sendMessage(msg);

        setState(STATE_CONNECTED);
    }

    public synchronized void stop() {
        deleteConnectThread();
        deleteConnectedThread();

        setState(STATE_NONE);
    }

    public void write(byte[] out, boolean flush) {
        // Create temporary object
        ConnectedThread r;
        // Synchronize a copy of the ConnectedThread
        synchronized (this) {
            if (mState != STATE_CONNECTED) return;
            r = mConnectedThread;
        }
        r.write(out, flush);
    }

    private void connectionFailed() {
        mHandler.obtainMessage(MESSAGE_CONNECTION_FAILED).sendToTarget();
        BluetoothScanndyService.this.start();
    }

    private void connectionLost() {
        mHandler.obtainMessage(MESSAGE_CONNECTION_LOST).sendToTarget();
        BluetoothScanndyService.this.start();
    }

    private class ConnectThread extends Thread {
        private final BluetoothSocket mmSocket;
        private final BluetoothDevice mmDevice;

        public ConnectThread(BluetoothDevice device) {
            mmDevice = device;

            BluetoothSocket tmp = null;
            try {
                tmp = device.createRfcommSocketToServiceRecord(SPP_UUID);
            } catch (IOException e) {
            }
            mmSocket = tmp;
        }

        public void run() {
            setName("ConnectThread");

            mAdapter.cancelDiscovery();

            try {
                mmSocket.connect();
            } catch (IOException e) {
                try {
                    mmSocket.close();
                } catch (IOException e2) {
                }
                connectionFailed();
                return;
            }

            synchronized (BluetoothScanndyService.this) {
                mConnectThread = null;
            }

            connected(mmSocket, mmDevice);
        }

        public void cancel() {
            try {
                mmSocket.close();
            } catch (IOException e) {
            }
        }
    }

    private class ConnectedThread extends Thread {
        private final BluetoothSocket mmSocket;
        private final InputStream mmInStream;
        private final OutputStream mmOutStream;

        public ConnectedThread(BluetoothSocket socket) {
            mmSocket = socket;
            InputStream tmpIn = null;
            OutputStream tmpOut = null;

            try {
                tmpIn = socket.getInputStream();
                tmpOut = socket.getOutputStream();
            } catch (IOException e) {
            }

            mmInStream = tmpIn;
            mmOutStream = tmpOut;
        }

        public void run() {
            byte[] buffer = new byte[1024];
            int bytes;
            String s = "";

            while (true) {
                try {
                    // Read from the InputStream
                    bytes = mmInStream.read(buffer);
    				for (int i = 0; i < bytes; ++i)
    					s += (char)buffer[i];

    				// 4, 1, 4, 1, n, 8
    				// [msgId], [flag (Q / R)], [length of payload], [reserved], [payload], [8 crc32 digits]
    				if (s.length() >= 18) {
    					if (s.charAt(4) == 'R' || s.charAt(4) == 'Q') {
    		        		int len = Integer.parseInt(Converter.hex2str(s.substring(5, 9)));
    						if (s.length() == len + 18)
    						{
    							Message msg = mHandler.obtainMessage(MESSAGE_READ);
    					        Bundle bundle = new Bundle();
    					        bundle.putString(SCOMP, s);
    					        msg.setData(bundle);
    					        mHandler.sendMessage(msg);
    					        s = "";
    						}
    					}
    					else
    						s = "";
    				}
                } catch (IOException e) {
                    connectionLost();
                    break;
                }
            }
        }

        public void write(byte[] buffer, boolean flush) {
            try {
                mmOutStream.write(buffer);
                if (flush)
                	mmOutStream.flush();
                
                Log.e("BTWrite", new String(buffer));
            } catch (IOException e) {
            }
        }

        public void cancel() {
            try {
    			mmInStream.close();
    			mmOutStream.close();
                mmSocket.close();
            } catch (IOException e) {
            }
        }
    }
}