package de.panmobil.scompdemo;

import java.util.zip.CRC32;

public class ScompResponseBuilder {

	private static final String BEEP = "beep";
	private static final String PRINT = "print";
	private static final String CLRSCR = "clrscr";
	private static final String BARSCAN = "barscan";
    private static final String ALL = "all";
    
    /*
     * A Scomp Msg structure: ssssTnnnnRp....pCCCCCCCC
     * see scomp-documentation for more info
     */

	/*
	 * build a default OK-Response Message
	 * parameters:
	 * msgid: a String, of a decimal msgid, you can change this in integer to handle it directly as integer
	 * at moment msgid have to 0010 (10)
	 */
	public static String buildOk(String msgid) {
		String n = fillWithNull(Integer.toHexString(2));	// gives the length of payload and fills it with '0' if the len isn't 4
		String okResponse = msgid + "R" + n + "#" + "ok";

		return okResponse + calcCrc(okResponse);
	}
	
	/*
	 * build a default Beep-Request Message
	 * parameters:
	 * msgid:	a String, of a decimal msgid, you can change this in integer to handle it directly as integer
	 * 			at moment msgid have to 0010 (10)
	 * freq:	frequenz of the beep tone
	 * ms:		milliseconds of the beep duration
	 */
	public static String buildBeep(String msgid, int freq, int ms) {
		String mid = "";
		if (msgid == null) {
			mid = fillWithNull(str2Hex(String.valueOf(MainActivity.lastMsgId)));
		} else {
			mid = msgid;
		}
		return buildBasicMsg(mid, BEEP + ":" + freq + "," + ms);
	}
	
	/*
	 * build a default Barscan-Request Message
	 * parameters:
	 * msgid:	a String, of a decimal msgid, you can change this in integer to handle it directly as integer
	 * 			at moment msgid have to 0010 (10)
	 * t:		time how long a scanndy should try to scan a barcode
	 */
	public static String buildBarscan(String msgid, int t) {
		String mid = "";
		if (msgid == null) {
			mid = fillWithNull(str2Hex(String.valueOf(MainActivity.lastMsgId)));
		} else {
			mid = msgid;
		}
		return buildBasicMsg(mid, BARSCAN + ":" + t);
	}

	/*
	 * build a special Print-on-Screen-Request Message for specially scanndy devices
	 * parameters:
	 * msgid:	a String, of a decimal msgid, you can change this in integer to handle it directly as integer
	 * 			at moment msgid have to 0010 (10)
	 * text:	String-text which should show on a screen of a scanndy device
	 */
	public static String buildPrintOnScreen(String msgid, String text) {
		String mid = "";
		if (msgid == null) {
			mid = fillWithNull(str2Hex(String.valueOf(MainActivity.lastMsgId)));
		} else {
			mid = msgid;
		}
		return buildBasicMsg(mid, PRINT + ":" + text);
	}
	
	/*
	 * build a special Clear-Screen-Request Message for specially scanndy devices
	 * parameters:
	 * msgid:	a String, of a decimal msgid, you can change this in integer to handle it directly as integer
	 * 			at moment msgid have to 0010 (10)
	 */
	public static String buildClearScreen(String msgid) {
		String mid = "";
		if (msgid == null) {
			mid = fillWithNull(str2Hex(String.valueOf(MainActivity.lastMsgId)));
		} else {
			mid = msgid;
		}
		return buildBasicMsg(mid, CLRSCR + ":" + ALL);
	}
	
	/*
	 * for all other messages (and in the future) to create your own Message
	 * parameters:
	 * msgid:	a String, of a decimal msgid, you can change this in integer to handle it directly as integer
	 * 			at moment msgid have to 0010 (10)
	 * payload: the payload what should send to the device,
	 * 			e.g. beep:350,500
	 */
	public static String buildBasicMsg(String msgid, String payload) {
		String mid = fillWithNull(Integer.toHexString((msgid != null) ? Integer.valueOf(msgid) : MainActivity.lastMsgId));

		String command = mid + "Q" + fillWithNull(Integer.toHexString(payload.length())) + "#" + payload;

		return command + calcCrc(command);
	}
	
	/*
	 * converts a String to hex
	 */
	public static String str2Hex(String text) {
		char[] c = text.toCharArray();
		
		StringBuffer hex = new StringBuffer();
		
		for ( int i = 0; i < c.length; i++ )
			hex.append(Integer.toHexString(Integer.valueOf(c[i])));

		return hex.toString();
	}
	
	/*
	 * fills a string of decimal numbers with 0 so that a length of it get 4
	 * e.g. 1B = 001B
	 */
	public static String fillWithNull(String n) {
		String t = "";
		if (n.length() < 4) {
			for (int i = 0; i < 4 - n.length(); i++) {
				t += "0";
			}
			t += n;
		}
		return t;
	}
	
	/*
	 * calculates the CRC-32 of a String
	 */
	public static String calcCrc(String s) {
		// create a crc from the scans
		CRC32 c = new CRC32();
		c.update(s.getBytes());
		String res = Long.toHexString(c.getValue());
		while (res.length() < 8)
			res = "0" + res;
		return res;
	}
}
