#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>

#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <sys/time.h>
#include <sys/types.h>

#include "scomp.h"

#define USE_READLINE


static char *tty_name = "/dev/ttyS0";
static int tty = -1;

static int ttySetRaw( int fd, struct termios *prevTermios )
{
    struct termios t;

    if ( tcgetattr( fd, &t ) == -1 )
        return -1;
    if ( prevTermios != NULL )
        *prevTermios = t;
    t.c_lflag &= ~( ICANON | ISIG | IEXTEN | ECHO );
                        /* Noncanonical mode, disable signals, extended
                           input processing, and echoing */
    t.c_iflag &= ~( BRKINT | ICRNL | IGNBRK | IGNCR | INLCR |
                      INPCK | ISTRIP | IXON | PARMRK );
                        /* Disable special handling of CR, NL, and BREAK.
                           No 8th-bit stripping or parity error handling.
                           Disable START/STOP output flow control. */
    t.c_oflag &= ~OPOST;     /* Disable all output processing */
    t.c_cc[VMIN] = 1;        /* Character-at-a-time input */
    t.c_cc[VTIME] = 0;       /* with blocking */
    if ( tcsetattr( fd, TCSAFLUSH, &t ) == -1 )
        return -1;
    return 0;
}


static void tty_reopen( void )
{
	close( tty );
	tty = -1;
	while ( 0 > tty )
	{
		tty = open( tty_name, O_RDWR );
		if ( 0 > tty )
		{
			fprintf( stderr, "waiting for tty ...\r" );
			sleep( 1 );
		}
	}
	ttySetRaw( tty, NULL );
	fprintf( stderr, "\n" );
	return;
}


static int timeDelta( struct timeval *st, struct timeval *end )
{
	int ds = end->tv_sec - st->tv_sec;
	int dus = end->tv_usec - st->tv_usec;

	if ( ds < 0 )
		ds += 86400;
	if ( dus < 0 )
	{
		dus += 1e6;
		ds--;
	}
	return ds * 1000 + dus / 1000;
}

static int sndcb( void *data, int size, int to )
{
	int n;
	fd_set fds;
	struct timeval selto;

	selto.tv_sec = to / 1000;
	selto.tv_usec = to % 1000 * 1000;
	FD_ZERO( &fds );
	FD_SET( tty, &fds );
	n = select( tty + 1, NULL, &fds, NULL, &selto );
	if ( 0 < n )
		n = write( tty, data, size );
 	return n;
}


static int rcvcb( void *data, int size, int to )
{
	int i;
	int n = 0;
	struct timeval start, now;
	struct timeval selto;
	fd_set fds;

	gettimeofday( &start, NULL );
	do
	{
		selto.tv_sec = to / 1000;
		selto.tv_usec = to % 1000 * 1000;
		FD_ZERO( &fds );
		FD_SET( tty, &fds );
		i = select( tty + 1, &fds, NULL, NULL, &selto );
		if ( 1 > i )
			return i;
		i = read( tty, (char *)data + n, size - n );
		if ( 0 > i )
			return i;
		if ( 0 == i )
		{
			fprintf( stderr, "EOF on TTY\n" );
			tty_reopen();
		}
		n += i;
		gettimeofday( &now, NULL );
		to -= timeDelta( &start, &now );
	}
	while ( n < size && 0 < to );
//((char*)data)[n] = '\0'; fprintf( stderr, "%s", (char*)data );
	return n;
}


static int handleRequest( const char *rbuf, int rlen, char *sbuf, int *slen )
{
	int res = 0;

	if ( 0 < rlen )
	{
		if ( 0 == strncmp( rbuf, "keydata:", 8 ) )
			res = rbuf[8];
		printf( "q: %s\n", rbuf );
		int n = snprintf( sbuf, *slen, "ok" );
		if ( *slen - 1 < n || 0 > n )
			res = -1;
		*slen = n;
		printf( "r: %s\n", sbuf );
	}
	else
		*slen = 0;
	return res;
	(void)rlen;
}


static int serve( void )
{
	int rlen, slen;
	int seq;
	int type;
	int res = 0;
	char buf[16000];

	rlen = sizeof buf - 1;
	slen = sizeof buf - 1;
	res = ScompRecv( buf, &rlen, &seq, &type, 50 );
	switch ( res )
	{
	case SCOMP_ERR_OK:
		if ( SCOMP_REQUEST == type )
		{
			int fnc = 0;
			buf[rlen] = '\0';
			fnc = handleRequest( buf, rlen, buf, &slen );
			if ( 0 <= fnc )
			{
				ScompSendResponse( buf, slen, seq );
				switch ( fnc )
				{
				case 'T': sprintf( buf, "barscan:2500" ); break;
				case '^': sprintf( buf, "rfidscan:tid" ); break;
				case 'D': sprintf( buf, "rfidscan:usr,0,4" ); break;
				case 'O': sprintf( buf, "rfidwrite:usr,0,%08X", (unsigned)time( NULL ) ); break;
				}
				printf( "Q: %s\n", buf );
				rlen = sizeof buf -1;
				ScompExch( buf, strlen( buf ), buf, &rlen, 5000 );
				buf[rlen] = '\0';
				printf( "R: %s\n", buf );
			}
			else
			{
				; // internal error
			}
		}
		else
		{
			; 	// drop unsolicited non-requests
		}
		break;
	case SCOMP_ERR_TIMEOUT:
		break; 	// silent timeout
	case SCOMP_ERR_CRC:
		ScompSendResponse( "error:crc", 9, seq );
		// fall-through!
	default:
		fprintf( stderr, "receive error: %s\n", ScompStrErr( res ) );
		break;
	}
	return res;
}


static char *trim( char *s )
{
	for ( char *p = s; *p; ++p )
	{
		if ( '\r' == *p || '\n' == *p )
		{
			*p = '\0';
			break;
		}
	}
	return s;
}

static int act( void )
{
	int n;
	char buf[4000];
#ifdef USE_READLINE
	if ( isatty( STDIN_FILENO ) )
	{
		char *line;
		line = readline( "\rQ: " );
		if ( !line )
		{
			fprintf( stderr, "EOF on stdin!\n" );
			exit( EXIT_FAILURE );
		}
		strncpy( buf, line, sizeof buf - 1);
		buf[sizeof buf - 1] = '\0';
		free( line );
		if ( *buf )
			add_history( buf );
	}
	else
#endif
	{
		n = read( STDIN_FILENO, buf, sizeof buf );
		if ( 0 == n )
		{
			fprintf( stderr, "EOF on stdin!\n" );
			exit( EXIT_SUCCESS );
		}
		if ( 0 > n )
			return -1;
		buf[n] = '\0';
		trim( buf );
	}
	n = sizeof buf;
	if ( 0 < strlen( buf ) )
	{
		ScompExch( buf, strlen( buf ), buf, &n, 5000 );
		buf[n] = '\0';
		printf( "R: %s\n", buf );
	}
	return 0;
}


int main( int argc, char *argv[] )
{
	fd_set fds;
	ScompOpt_t opt;

	if ( 1 < argc )
		tty_name = argv[1];
	tty_reopen();

	opt.i = 1;
	ScompSetOption( SCOMP_OPT_USE_CRC, opt );
	opt.cb = sndcb;
	ScompSetOption( SCOMP_OPT_SNDCB, opt );
	opt.cb = rcvcb;
	ScompSetOption( SCOMP_OPT_RCVCB, opt );

#ifdef USE_READLINE
	if ( isatty( STDIN_FILENO ) )
	{
		struct termios t;
	    tcgetattr( STDIN_FILENO, &t );
	    t.c_lflag &= ~( ICANON ); // noncanonical mode
	    tcsetattr( STDIN_FILENO, TCSAFLUSH, &t );
	}
#endif

	while ( 1 )
	{
		printf( "Q: " );
		fflush( stdout );
		FD_SET( STDIN_FILENO, &fds );
		FD_SET( tty, &fds );
		if ( 0 >= select( tty + 1, &fds, NULL, NULL, NULL ) )
		{
			if ( EINTR != errno )
				break;
		}
		else if ( FD_ISSET( STDIN_FILENO, &fds ) )
		{
			act();
		}
		else if ( FD_ISSET( tty, &fds ) )
		{
			printf( "\r" );
			fflush( stdout );
			serve();
		}
	}

	perror( "select" );
    return 0;
}

/* EOF */
